(** A very fast demand-driven constraint solver *)
module Make (Var: Hashtbl.HashedType) (Dom: Lattice.S) = 
struct
  module HT = Hash.Make(Var)
  type 'a table    = 'a HT.t
  type variable    = Var.t
  type domain      = Dom.t

  type assignment  = variable -> domain
  type assignment' = domain table (* data structure representation of an assignment *)
  type rhs         = assignment -> domain (* rhs of the constraint in functional form *)
  type lhs         = variable (* system variable *)
  type constrain   = lhs * rhs  (* a single constraint: lhs \sqsupseteq rhs *)
  type system      = lhs -> rhs list (* maps variables to it's set of constraints *)

  type solution    = assignment'

  let initstate	   = ref (fun (x : variable) -> Dom.bot ())

  (** solves a constraint system in a demand-driven fashion starting from the
    * initialvars *)
  let solve (system: system) (initialvars: variable list): solution =
    (* sigma will be the solution, it is initially empty; new variables are 
     * initialized to bottom as they occur. *)
    let sigma: assignment' = HT.create 113 (Dom.bot ()) in
    (* infl captures the dynamic dependencies: infl x is the set of constaints
     * that should be recomputed whenever x changes *)
    let infl: constrain list table = HT.create 113 [] in
    (* the worklist of rh-sides that should be considered for each variable! *)
    let todo: rhs list table = HT.create 113 [] in

    let rec constrain_one_var (x: variable) = 
      let old_state = if HT.mem sigma x 
                      then HT.find sigma x 
                      else !initstate x in
      let local_state = ref old_state in 
      (* here we find the set of constraints that should be considered *)
      let rhsides = 
        let notnew = HT.mem sigma in 
          if notnew x then
            (* look up the constraints that need processing  *)
            let temp = HT.find todo x in HT.remove todo x; temp
          else begin
            (* initialize the new variable to bottom *)
            (*HT.add sigma x (Dom.bot ());*)
            HT.add sigma x (!initstate x);
            (* return all the constraints for that variable *)
            system x
          end
      in
      let apply_one_constraint (f: rhs) =
        let nls = f (eval (x,f)) in 
          local_state := Dom.join !local_state nls
      in
        List.iter apply_one_constraint rhsides;
        if not (Dom.equal !local_state old_state) then begin
          (* if the state has changed we update it *)
          HT.replace sigma x !local_state;
          (* the following adds the rhs of the influenced constraints to our
           * todo list and immediately solves for their corresponding variables *)
          let influenced_vars = ref [] in
          let collect_influence (y,f) = 
            HT.replace todo y (f :: HT.find todo y);
            influenced_vars := y :: !influenced_vars
          in
            List.iter collect_influence (HT.find infl x);
            List.iter constrain_one_var !influenced_vars;
        end

    and eval (c: constrain) (v: variable) =
      (* demand driven computation of the variable *)
      constrain_one_var v;
      (* add c to the set of constraints that are influenced by v *)
      HT.replace infl v (c :: HT.find infl v);
      (* finally forward the value of v *)
      HT.find sigma v 

    in
      List.iter constrain_one_var initialvars;
      sigma
end 
